/*
 * DynamicGeneCircuit.cpp
 *
 *  Created on: May 2, 2012
 *      Author: linh
 */

#include <math.h>
#include "DynamicGeneCircuit.h"
#include "GA/GASimpleGA.h"			// we're going to use the simple GA
#include "GA/GA1DArrayGenome.h" 	// and the 2D binary string genome
#include "GA/GABin2DecGenome.h"


Bound::Bound (double l, double u) {
	lower = l;
	upper = u;
}

Component::Component() {
	sub_component_list.clear();
	inputs.clear();
	outputs.clear();
	linkages.clear();
	variant_id = 1;
}

Component::~Component() {
	// Do nothing
}

void Component::ClearAll() {
	// TODO:
}

void Component::operator << (ostream& out) {
	for (unsigned int i = 0; i < sub_component_list.size(); i++) {
		sub_component_list[i]->operator << (out);
		out << endl;
	}
}

void Component::Linearize(vector<MoleculePool*>& pool_list) {
	if (sub_component_list.empty()) {
		pool_list.push_back((MoleculePool*)this);
	}
	else
		for (unsigned int i = 0; i < sub_component_list.size(); i++)
			sub_component_list[i]->Linearize(pool_list);
}

MoleculePool::MoleculePool() {
	min_concentration = UNKNOWN;
	max_concentration = UNKNOWN;
	concentration = 0;
	min_concentration_list.clear();
	max_concentration_list.clear();
	current_interval_index.clear();
	sub_component_list.clear();
	children.clear();
	parents.clear();
}

MoleculePool::~MoleculePool() {
	// Do nothing
}

void MoleculePool::operator << (ostream& out) {
	molecule->operator << (out);
	out << "\t variant id = \t" << variant_id << endl;
}

double MoleculePool::SteadyStateChange(const PartDatabase* pdb) {
	int type = molecule->getType();
	double old_concentration = concentration;
	switch (type) {
		case LIGAND: {
			if (this->parents.size() > 0) { // Ligand is produced by a metabolism process, we assume that the concentration = 1/2 concentration of enzyme
				// TODO: simulate protein -> ligand
				concentration = this->parents[0]->concentration*0.5;
			}
			break;
		}
		case m_RNA: {
			// mRNA is produced by the transcription process at different genes
			concentration = pdb->getPromoterStrength(molecule->name, this->variant_id);
			for (unsigned int i = 0; i < parents.size(); i++) {
				TranscriptionRegulation tr = pdb->getPromoterRegulation(parents[i]->molecule->name,molecule->name);
				if (tr.regulator_type == ACTIVATORY)
					concentration *= (1 - 1/(1 + tr.binding_affinity*pow(parents[i]->concentration, tr.cooperativity)));
				else // PROHIBITORY
					concentration /= (1 + tr.binding_affinity*pow(parents[i]->concentration, tr.cooperativity));
			}
			concentration += pdb->getPromoterBasal(molecule->name, this->variant_id);
			break;
		}
		case PROTEIN: {
			// Unbound protein is produced by the translation process
			concentration = 0;
			bool is_output_protein = false;
			for (unsigned int i = 0; i < parents.size(); i++) {
				if (parents[i]->molecule->getType() == m_RNA || parents[i]->molecule->getType() == RNA_COMPLEX)
					concentration += parents[i]->concentration;
				else if (parents[i]->molecule->getType() == POOL) { // A special case for the output when a protein (e.g. gfp) is produced from a pool
					concentration = parents[i]->concentration;
					is_output_protein = true;
					break;
				}
				else if (parents[i]->molecule->getType() != LIGAND) {
					cout << "error in simulation: PROTEIN contains: " << parents[i]->molecule->getType() << " which is produced neither m_RNA nor mRNA_COMPLEX nor LIGAND" << endl;
					return UNKNOWN;
				}
			}
			if (is_output_protein)
				break;
			concentration /= pdb->getDegradationRate(molecule->name);
			for (unsigned int i = 0; i < parents.size(); i++)
				if (parents[i]->molecule->getType() == LIGAND) {
					LigandRegulation lr = pdb->getLigandRegulation(parents[i]->molecule->name, molecule->name);
					// ligands always repress the protein production
					concentration *= (pow(lr.dissociation, lr.cooperativity)/(pow(lr.dissociation, lr.cooperativity) + pow(parents[i]->concentration, lr.cooperativity)));
				}
			break;
		}
		case RNA_COMPLEX: {
			concentration = 1;
			for (unsigned int i = 0; i < parents.size(); i++)
				concentration *= parents[i]->concentration;
			break;
		}
		case PROTEIN_COMPLEX: {
			concentration = 1;
			for (unsigned int i = 0; i < parents.size(); i++)
				concentration *= parents[i]->concentration;
			break;
		}
		case LIGAND_PROTEIN_COMPLEX: {
			// Bound protein is produced by translation process + binding of inducers
			// TODO: Currently, only one protein + one ligand are considered
			string protein_name, ligand_name;
			int protein_position, ligand_position;
			for (unsigned int i = 0; i < parents.size(); i++) {
				if (parents[i]->molecule->getType() == PROTEIN) {
					protein_name = parents[i]->molecule->name;
					protein_position = i;
				}
				else if (parents[i]->molecule->getType() == LIGAND) {
					ligand_name = parents[i]->molecule->name;
					ligand_position = i;
				}
				else {
					cout << "error in simulation" << endl;
					return UNKNOWN;
				}
			}
			// ligands always activate the ligand-protein complex production
			LigandRegulation lr = pdb->getLigandRegulation(parents[ligand_position]->molecule->name, parents[protein_position]->molecule->name);
			concentration = parents[protein_position]->concentration*(pow(parents[ligand_position]->concentration, lr.cooperativity)/(pow(lr.dissociation, lr.cooperativity) + pow(parents[ligand_position]->concentration, lr.cooperativity)));
			break;
		}
		case POOL: {
			concentration = 0;
			for (unsigned int i = 0; i < parents.size(); i++)
				concentration += parents[i]->concentration;
			break;
		}
	}

	return (concentration - old_concentration)*(concentration - old_concentration);
}

DynamicGeneCircuit::DynamicGeneCircuit() {
	// TODO Auto-generated constructor stub
}

DynamicGeneCircuit::DynamicGeneCircuit(Module* gene_circuit_graph) {
	root_component = new Component;

	int n = gene_circuit_graph->NodeCount();
	root_component->sub_component_list.resize(n);
	for (int i = 0; i < n; i++)
		root_component->sub_component_list[i] = BioNetNode2MoleculePool(gene_circuit_graph->GetNodeAttr(i));
	for (int i = 0; i < n; i++)
		for (int j = 0; j < n; j++)
			if ((i != j) && gene_circuit_graph->GetEdgeAttr(i,j) != NULL) {
				((MoleculePool*)root_component->sub_component_list[i])->children.push_back((MoleculePool*)root_component->sub_component_list[j]);
				((MoleculePool*)root_component->sub_component_list[j])->parents.push_back((MoleculePool*)root_component->sub_component_list[i]);
			}

	// No linkages since this will return an elementary component
	root_component->linkages.empty();
	// inputs
	for (unsigned int i = 0; i < gene_circuit_graph->inputs.size(); i++)
		root_component->inputs.push_back((MoleculePool*)root_component->sub_component_list[gene_circuit_graph->inputs[i]]);
	// outputs
	for (unsigned int i = 0; i < gene_circuit_graph->outputs.size(); i++)
		root_component->outputs.push_back((MoleculePool*)root_component->sub_component_list[gene_circuit_graph->outputs[i]]);
}

DynamicGeneCircuit::~DynamicGeneCircuit() {
	RecursiveClear(root_component);
}

void DynamicGeneCircuit::RecursiveClear(Component* component_node) {
	for(unsigned int i = 0; i < component_node->sub_component_list.size(); i++)
		RecursiveClear(component_node->sub_component_list[i]);
	delete component_node;
}

void DynamicGeneCircuit::PrintOut(ostream* f) {
	root_component->operator <<(*f);
}

SimulationStatus DynamicGeneCircuit::SimulateSteadyState(const PartDatabase* pdb, const ValueMatrix& input_values, ValueMatrix& output_values, double threshold_value, int max_iteration) {

	// Resize the output value matrix
	output_values.resize(input_values.size());
	for (unsigned int i = 0; i < output_values.size(); i++)
		output_values[i].resize(root_component->outputs.size());

	bool over_time = false;
	vector<MoleculePool*> molecule_list;
	root_component->Linearize(molecule_list);
	int n = molecule_list.size();
	for (int i = 0; i < n; i++)
		molecule_list[i]->best_concentration_list.clear();

	for (unsigned int condition = 0; condition < input_values.size(); condition++) {
		// Assign the input concentration for inducers
		for (unsigned int i = 0; i < root_component->inputs.size(); i++)
			root_component->inputs[i]->concentration = input_values[condition][i];

		// Simulate here
		int count = 0;
		double error_tmp_max;
		do {
			error_tmp_max = 0;
			for (int i = 0; i < n; i++) {
				double diff = molecule_list[i]->SteadyStateChange(pdb);
				if (diff > error_tmp_max)
					error_tmp_max = diff;
			}
			count++;
		}
		while (error_tmp_max > threshold_value && count < max_iteration);
		if (count >= max_iteration) {
			cout << "Simulation runs over time" << endl;
			over_time = true;
		}
		// debug
		//for (int i = 0; i < n; i++)
		//	cout << molecule_list[i]->molecule->name << "\t\t" << molecule_list[i]->concentration << endl;
		//cout << "---------------" << endl;
		// end debug
		for (unsigned int i = 0; i < root_component->outputs.size(); i++)
			output_values[condition][i] = root_component->outputs[i]->concentration;
	}

	return (over_time)?OVER_TIME:CONVERGENCE;
}

DynamicGeneCircuit* DummyForGA::dynamic_gene_circuit = NULL;
const PartDatabase* DummyForGA::part_database = NULL;
//const DesignProblem* DummyForGA::design_problem = NULL;
const ValueMatrix* DummyForGA::inputs = new ValueMatrix;
const ValueMatrix* DummyForGA::outputs = new ValueMatrix;
vector<MoleculePool*> DummyForGA::promoter_list;
IdList DummyForGA::number_of_mutants_list;

void DynamicGeneCircuit::ExhaustiveSearch(const PartDatabase* pdb, const ValueMatrix& input_values, const ValueMatrix& output_values) {
	// Find all promoters that have more than one mutant

	vector<MoleculePool*> molecule_list;
	root_component->Linearize(molecule_list);
	DummyForGA::promoter_list.clear();
	DummyForGA::number_of_mutants_list.clear();
	for (unsigned int i = 0; i < molecule_list.size(); i++)
		if (molecule_list[i]->molecule->getType() == m_RNA) {
			int number_of_mutants = pdb->getNumberOfMutants(molecule_list[i]->molecule->name);
			if (number_of_mutants > 1) {
				DummyForGA::promoter_list.push_back(molecule_list[i]);
				DummyForGA::number_of_mutants_list.push_back(number_of_mutants);
			}
			// TEST HERE
			//cout << molecule_list[i]->molecule->name << "\t" << number_of_mutants << endl;
		}
	DummyForGA::dynamic_gene_circuit = this;
	DummyForGA::part_database = pdb;
	DummyForGA::inputs = &input_values;

	// Brute-force search
	IdList best_mutant (DummyForGA::promoter_list.size());
	for (unsigned int i = 0; i < DummyForGA::promoter_list.size(); i++)
		(DummyForGA::promoter_list[i])->variant_id = 1;
	bool terminate = false;
	double error = 1e10;

	int count = 0;
	do {
		// TEST HERE
		count++;
		if (count % 1000 == 0)
			cout << count << endl;

		double error_tmp = 0;
		bool good_simulation = true;

		ValueMatrix output_value_tmp;
		good_simulation = good_simulation && (DummyForGA::dynamic_gene_circuit->SimulateSteadyState(DummyForGA::part_database, input_values, output_value_tmp) == CONVERGENCE);
		for (unsigned int i = 0; i < output_value_tmp.size(); i++)
			for (unsigned int j = 0; j < output_value_tmp[i].size(); j++)
				error_tmp += pow(output_value_tmp[i][j] - output_values[i][j],2);
		// Estimate error
		if (good_simulation && error_tmp < error) {
			// Update the best solution
			//cout << error_tmp << endl;
			error = error_tmp;
			for (unsigned int i = 0; i < best_mutant.size(); i++)
				best_mutant[i] = (DummyForGA::promoter_list[i])->variant_id;
		}
		terminate = true;
		for (unsigned int i = 0; i < DummyForGA::promoter_list.size(); i++)
			if ((DummyForGA::promoter_list[i])->variant_id < DummyForGA::number_of_mutants_list[i]) {
				(DummyForGA::promoter_list[i])->variant_id++;
				terminate = false;
				break;
			}
			else
				(DummyForGA::promoter_list[i])->variant_id = 1;
	}
	while (!terminate);

	// Simulate again
	for (unsigned int i = 0; i < DummyForGA::promoter_list.size(); i++)
		(DummyForGA::promoter_list[i])->variant_id = best_mutant[i];
	ValueMatrix output_values_tmp;
	SimulateSteadyState(DummyForGA::part_database, *(DummyForGA::inputs), output_values_tmp);
	for (unsigned int i = 0; i < output_values_tmp.size(); i++) {
		for (unsigned int j = 0; j < output_values_tmp[i].size(); j++)
			cout << output_values_tmp[i][j] << "(" << output_values[i][j] << ")" << "\t";
		cout << endl;
	}
}

void DynamicGeneCircuit::GASearch(const PartDatabase* pdb, const ValueMatrix& input_values, const ValueMatrix& output_values) {
	// Find all promoters that have more than one mutant

	vector<MoleculePool*> molecule_list;
	root_component->Linearize(molecule_list);
	DummyForGA::promoter_list.clear();
	DummyForGA::number_of_mutants_list.clear();
	for (unsigned int i = 0; i < molecule_list.size(); i++)
		if (molecule_list[i]->molecule->getType() == m_RNA) {
			int number_of_mutants = pdb->getNumberOfMutants(molecule_list[i]->molecule->name);
			if (number_of_mutants > 1) {
				DummyForGA::promoter_list.push_back(molecule_list[i]);
				DummyForGA::number_of_mutants_list.push_back(number_of_mutants);
			}
		}
	// Run GA here
	// Number of genes for GA
	//int length = 0;
	GABin2DecPhenotype map;
	for (unsigned int i = 0; i < DummyForGA::number_of_mutants_list.size(); i++)
		map.add(ceil(log2(DummyForGA::number_of_mutants_list[i])), 0, DummyForGA::number_of_mutants_list[i] - 1);

	DummyForGA::dynamic_gene_circuit = this;
	DummyForGA::part_database = pdb;
	//DummyForGA::design_problem = &design_problem;
	DummyForGA::inputs = &input_values;
	DummyForGA::outputs = &output_values;

	//GA1DArrayGenome<unsigned int> genome(length, DummyForGA::GAFitness);				// create a genome
	GABin2DecGenome genome(map, DummyForGA::GAFitness);
	GASimpleGA ga(genome);																// create the genetic algorithm
	ga.populationSize(20); // 200
	ga.nGenerations(5);	// 50
	ga.pMutation(0.01);
	ga.pCrossover(0.9);

	ga.evolve();
	//cout << ga.statistics() << endl;

	// Simulate again
	GAGenome g = ga.statistics().bestIndividual();
	//cout << ga.statistics().bestIndividual();
	for (unsigned int i = 0; i < DummyForGA::promoter_list.size(); i++)
		DummyForGA::promoter_list[i]->variant_id = ((GA1DArrayGenome<unsigned int>*)&g)->gene(i) % DummyForGA::number_of_mutants_list[i] + 1;
	ValueMatrix output_values_tmp;
	SimulateSteadyState(DummyForGA::part_database, *(DummyForGA::inputs), output_values_tmp);
	for (unsigned int i = 0; i < output_values_tmp.size(); i++) {
		for (unsigned int j = 0; j < output_values_tmp[i].size(); j++)
			cout << output_values_tmp[i][j]  << "\t";
		cout << endl;
	}
}

float DummyForGA::GAFitness(GAGenome& g) {
	GA1DArrayGenome<unsigned int> &genome = (GA1DArrayGenome<unsigned int> &) g;
	for (unsigned int i = 0; i < DummyForGA::promoter_list.size(); i++)
		(DummyForGA::promoter_list[i])->variant_id = genome.gene(i) % DummyForGA::number_of_mutants_list[i] + 1;

	ValueMatrix output_values;
	DummyForGA::dynamic_gene_circuit->SimulateSteadyState(DummyForGA::part_database, *(DummyForGA::inputs), output_values);
	double error = 0;
	for (unsigned int i = 0; i < output_values.size(); i++)
		for (unsigned int j = 0; j < output_values[i].size(); j++)
			error += pow(output_values[i][j] - (*DummyForGA::outputs)[i][j],2);
	//cout << 100 - error << endl;
	return 100 - error;
}

MoleculePool* BioNetNode2MoleculePool(BioNetNode* bio_node) {
	MoleculePool* tmp = new MoleculePool;
	tmp->molecule = (Molecule*)bio_node->component->clone();
	return tmp;
}
